<?php declare(strict_types=1);
require __DIR__ . '/head.php';

function pdo(): PDO {
    static $pdo = null;
    if ($pdo) return $pdo;
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
    $pdo = new PDO($dsn, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
    return $pdo;
}
function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }

/* Distinkta värden för dropdowns */
$distinct = [
    'product_type' => pdo()->query("SELECT DISTINCT product_type v FROM refurb_products ORDER BY v")->fetchAll(PDO::FETCH_COLUMN),
    'manufacturer' => pdo()->query("SELECT DISTINCT manufacturer v FROM refurb_products ORDER BY v")->fetchAll(PDO::FETCH_COLUMN),
    'grade'        => pdo()->query("SELECT DISTINCT grade v FROM refurb_products ORDER BY v")->fetchAll(PDO::FETCH_COLUMN),
];

/* Filters (GET) */
$filters = [
    'product_type'  => trim($_GET['product_type'] ?? ''),
    'manufacturer'  => trim($_GET['manufacturer'] ?? ''),
    'grade'         => trim($_GET['grade'] ?? ''),
    'min_price'     => trim($_GET['min_price'] ?? ''), // inkl moms
    'max_price'     => trim($_GET['max_price'] ?? ''), // inkl moms
];

/* Sortering (server-side) */
$sortCol = $_GET['sort'] ?? 'sku';
$sortDir = strtolower($_GET['dir'] ?? 'asc');
$sortDir = $sortDir === 'desc' ? 'DESC' : 'ASC';

$sortable = [
    'sku' => 'sku',
    'description' => 'description',
    'product_type' => 'product_type',
    'manufacturer' => 'manufacturer',
    'grade' => 'grade',
    'quantity' => 'quantity',
    // pris inkl moms som uttryck
    'price' => '(price_net_sek * (1 + vat_rate_percent/100))',
    'valid_until' => 'valid_until',
];
$orderExpr = $sortable[$sortCol] ?? 'sku';

/* WHERE */
$w = []; $p = [];
if ($filters['product_type'] !== '') { $w[]='product_type = :ptype'; $p[':ptype'] = $filters['product_type']; }
if ($filters['manufacturer'] !== '') { $w[]='manufacturer = :mfg';   $p[':mfg']   = $filters['manufacturer']; }
if ($filters['grade'] !== '')        { $w[]='grade = :grade';        $p[':grade'] = $filters['grade']; }

if ($filters['min_price'] !== '') {
    $w[] = '(price_net_sek * (1 + vat_rate_percent/100)) >= :pmin';
    $p[':pmin'] = (float)str_replace(',', '.', $filters['min_price']);
}
if ($filters['max_price'] !== '') {
    $w[] = '(price_net_sek * (1 + vat_rate_percent/100)) <= :pmax';
    $p[':pmax'] = (float)str_replace(',', '.', $filters['max_price']);
}
$where = $w ? ('WHERE '.implode(' AND ', $w)) : '';

/* Hämta ALLA rader (ingen pagination) */
$sql = "SELECT sku, description, product_type, manufacturer, grade, quantity,
               price_net_sek, vat_rate_percent, valid_until
        FROM refurb_products
        $where
        ORDER BY $orderExpr $sortDir";
$st = pdo()->prepare($sql);
foreach ($p as $k=>$v) $st->bindValue($k, $v);
$st->execute();
$rows = $st->fetchAll();
$total = count($rows);

/* Helpers */
function q(array $merge = []): string { $q = array_merge($_GET, $merge); return '?'.http_build_query($q); }
function sortLink(string $col, string $label){
    $curCol = $_GET['sort'] ?? 'sku';
    $curDir = strtolower($_GET['dir'] ?? 'asc');
    $nextDir = ($curCol === $col && $curDir === 'asc') ? 'desc' : 'asc';
    $classes = 'sort' . ($curCol === $col ? ' ' . $curDir : '');
    echo '<th><a class="'.h($classes).'" href="'.h(q(['sort'=>$col,'dir'=>$nextDir])).'">'.h($label).'</a></th>';
}
?>
<!doctype html>
<html lang="sv">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Prislista</title>
<style>
:root{
  --bg:#f8fafc;--card:#fff;--text:#0f172a;--muted:#475569;--line:#e2e8f0;--accent:#2563eb;
}
*{box-sizing:border-box}
body{margin:0;background:var(--bg);color:var(--text);font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu,"Helvetica Neue",Arial,sans-serif}
.container{max-width:1200px;margin:0 auto;padding:0 10px}

/* Sticky topbar med filter */
.topbar{ position:sticky; top:0; z-index:10; background:var(--bg); border-bottom:1px solid var(--line); }
.topbar-inner{ display:flex; align-items:center; gap:8px; padding:6px 0; min-height:32px }
.topbar h1{ font-size:1rem; margin:0 8px 0 0 }
.meta{ display:flex; gap:8px; align-items:center; flex-wrap:wrap }
.badge{ display:inline-block; background:#eef2ff; border:1px solid #c7d2fe; color:#3730a3; padding:.15rem .35rem; border-radius:999px; font-size:.75rem }
.btn{ display:inline-block; background:#e2e8f0; color:#0f172a; border:0; border-radius:7px; padding:.32rem .5rem; font-weight:700; cursor:pointer }

/* Filter inline i topbar */
.filters-inline{ display:flex; flex-wrap:wrap; gap:6px; padding:4px 0 6px 0; }
.filters-inline .field{ display:flex; align-items:center; gap:6px; background:var(--card); border:1px solid var(--line); border-radius:7px; padding:4px 6px; }
.filters-inline label{ font-size:.78rem; color:var(--muted) }
.filters-inline select, .filters-inline input{
  border:1px solid var(--line); border-radius:6px; background:#fff;
  padding:.28rem .4rem; font-size:.85rem; min-width:110px;
}
.filters-inline .narrow{ min-width: 70px; }

/* Tabell */
.card{ background:var(--card); border:1px solid var(--line); border-radius:10px; margin:8px 0 }
.table-wrap{ overflow:auto; max-height:calc(100vh - 170px); border-radius:10px }
table{ width:100%; border-collapse:separate; border-spacing:0; min-width:860px }
thead th{ position:sticky; top:0; background:#fff; border-bottom:1px solid var(--line); z-index:2 }
th,td{ padding:.45rem .5rem; border-bottom:1px solid var(--line); text-align:left; font-size:.93rem; vertical-align:top }
tbody tr:nth-child(odd){ background:#ffffff }
tbody tr:nth-child(even){ background:#f8fafc }
tbody tr:hover{ background:#eef2ff }
th a{ color:inherit; text-decoration:none }
th a.sort::after{ content:" ⇅"; opacity:.5 }
th a.sort.asc::after{ content:" ↑" }
th a.sort.desc::after{ content:" ↓" }

/* Små skärmar */
@media (max-width: 920px){
  .filters-inline .field{ padding:4px }
  .filters-inline select,.filters-inline input{ min-width: 90px }
}
</style>
</head>
<body>
<div class="container">

  <div class="topbar">
    <div class="topbar-inner">
      <h1>Prislista</h1>
      <div class="meta">
        <span class="badge">Totalt: <?= (int)$total ?> st</span>
        <a class="btn" href="<?= h(q()) ?>">Uppdatera</a>
        <a class="btn" href="?">Rensa</a>
      </div>
    </div>

    <!-- FILTER I LÅSTA DELEN -->
    <form class="filters-inline" method="get" id="filterForm">
      <div class="field">
        <label>Typ</label>
        <select name="product_type">
          <option value="">Alla</option>
          <?php foreach ($distinct['product_type'] as $v): ?>
            <option value="<?= h($v) ?>" <?= $filters['product_type']===$v?'selected':'' ?>><?= h($v) ?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="field">
        <label>Tillverkare</label>
        <select name="manufacturer">
          <option value="">Alla</option>
          <?php foreach ($distinct['manufacturer'] as $v): ?>
            <option value="<?= h($v) ?>" <?= $filters['manufacturer']===$v?'selected':'' ?>><?= h($v) ?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="field">
        <label>Grade</label>
        <select name="grade">
          <option value="">Alla</option>
          <?php foreach ($distinct['grade'] as $v): ?>
            <option value="<?= h($v) ?>" <?= $filters['grade']===$v?'selected':'' ?>><?= h($v) ?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="field">
        <label>Pris min (inkl)</label>
        <input class="narrow" type="text" name="min_price" value="<?= h($filters['min_price']) ?>" placeholder="">
      </div>
      <div class="field">
        <label>Pris max (inkl)</label>
        <input class="narrow" type="text" name="max_price" value="<?= h($filters['max_price']) ?>" placeholder="">
      </div>
    </form>
  </div>

  <div class="card table-wrap">
    <table>
      <thead>
        <tr>
          <?php sortLink('sku','SKU'); ?>
          <?php sortLink('description','Beskrivning'); ?>
          <?php sortLink('product_type','Produkttyp'); ?>
          <?php sortLink('manufacturer','Tillverkare'); ?>
          <?php sortLink('grade','Grade'); ?>
          <?php sortLink('quantity','Antal'); ?>
          <?php sortLink('price','Pris (inkl)'); ?>
          <?php sortLink('valid_until','Gäller t.o.m.'); ?>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($rows as $r): ?>
          <?php
            $gross = (float)$r['price_net_sek'] * (1.0 + ((float)$r['vat_rate_percent'] / 100.0));
            $grossRounded = (int)round($gross, 0); // hela kronor
          ?>
          <tr>
            <td><?= h((string)$r['sku']) ?></td>
            <td><?= h($r['description']) ?></td>
            <td><?= h($r['product_type']) ?></td>
            <td><?= h($r['manufacturer']) ?></td>
            <td><?= h($r['grade']) ?></td>
            <td><?= (int)$r['quantity'] ?></td>
            <!-- Pris utan mellanslag och utan 'kr' -->
            <td><?= $grossRounded ?></td>
            <td><?= $r['valid_until'] ? h($r['valid_until']) : '—' ?></td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>

</div>

<!-- Autosubmit för filter -->
<script>
(function(){
  const form = document.getElementById('filterForm');
  if (!form) return;

  // Dropdowns: skicka direkt vid ändring
  form.querySelectorAll('select').forEach(sel=>{
    sel.addEventListener('change', ()=> form.submit());
  });

  // Textfält (pris min/max): skicka med debounce efter 400 ms
  let t;
  form.querySelectorAll('input[type="text"]').forEach(inp=>{
    inp.addEventListener('input', ()=>{
      clearTimeout(t);
      t = setTimeout(()=> form.submit(), 400);
    });
    inp.addEventListener('keydown', (e)=>{ if(e.key==='Enter'){ e.preventDefault(); form.submit(); }});
  });
})();
</script>
</body>
</html>
