<?php declare(strict_types=1);
require __DIR__ . '/head.php';

function pdo(): PDO {
    static $pdo = null;
    if ($pdo) return $pdo;
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
    $pdo = new PDO($dsn, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::MYSQL_ATTR_LOCAL_INFILE => false,
    ]);
    return $pdo;
}

/* Hjälpfunktioner */
function lower(string $s): string {
    return function_exists('mb_strtolower') ? mb_strtolower($s, 'UTF-8') : strtolower($s);
}
function normalizeSpaces(string $s): string {
    $s = str_replace(["\xC2\xA0", "\t"], ' ', $s);
    $s = preg_replace('/[ ]{2,}/u', ' ', $s);
    return trim($s ?? '');
}
function normalizePriceToDecimal(string $s): ?string {
    $s = normalizeSpaces($s);
    $s = preg_replace('/\s*(kr|sek)\.?$/iu', '', $s); // ta bort "kr"/"SEK"
    $s = preg_replace('/\s+/u', '', $s);              // ta bort alla mellanslag
    if (strpos($s, '.') !== false && strpos($s, ',') !== false) {
        // anta . = tusen, , = decimal
        $s = str_replace('.', '', $s);
        $s = str_replace(',', '.', $s);
    } else {
        $s = str_replace(',', '.', $s);
    }
    if ($s === '' || !preg_match('/^\d+(\.\d+)?$/', $s)) return null;
    return number_format((float)$s, 2, '.', '');
}
function looksLikeHeader(array $cols): bool {
    $n = array_map(fn($c) => lower(trim((string)$c)), $cols);
    return in_array('nr', $n, true) && in_array('beskrivning', $n, true) && in_array('produkttyp', $n, true);
}
function mapHeader(array $cols): array {
    $map = [];
    foreach ($cols as $i => $raw) {
        $k = lower(trim((string)$raw));
        $k = preg_replace('/\s+/u', ' ', $k);
        if ($k !== '') $map[$k] = $i;
    }
    return $map;
}
/** Minimikrav: Nr, Beskrivning, Produkttyp, Tillverkare + pris (ÅF-Pris eller AF-Pris) */
function hasCoreFields(array $map): bool {
    $hasPrice = isset($map['åf-pris']) || isset($map['af-pris']);
    return isset($map['nr'], $map['beskrivning'], $map['produkttyp'], $map['tillverkare']) && $hasPrice;
}
/** Diagnostik av hoppade rader */
class SkipLog {
    public array $count = [];
    public array $examples = [];
    public int $maxExamples = 10;
    public function add(string $reason, array $row): void {
        $this->count[$reason] = ($this->count[$reason] ?? 0) + 1;
        if (!isset($this->examples[$reason])) $this->examples[$reason] = [];
        if (count($this->examples[$reason]) < $this->maxExamples) {
            $snippet = implode(' | ', array_slice(array_map(fn($c)=> normalizeSpaces((string)$c), $row), 0, 8));
            $this->examples[$reason][] = $snippet;
        }
    }
}
/** Avrunda brutto enligt vald strategi + steg */
function round_price(float $amount, string $strategy, int $unit): float {
    $unit = max(1, $unit);
    $x = $amount / $unit;
    switch ($strategy) {
        case 'nearest': $x = round($x); break;
        case 'ceil':    $x = ceil($x);  break;
        case 'floor':   $x = floor($x); break;
        case 'none':
        default:        return $amount;
    }
    return $x * $unit;
}

$feedback = [];
$report   = null;
$debug    = isset($_GET['debug']);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Parametrar från formulär
    $discountPercent   = isset($_POST['discount_percent']) ? (float)str_replace(',', '.', $_POST['discount_percent']) : 0.0;
    $discountFixed     = isset($_POST['discount_fixed'])   ? (float)str_replace(',', '.', $_POST['discount_fixed'])   : 0.0;
    $shippingPerUnit   = isset($_POST['shipping_per_unit'])? (float)str_replace(',', '.', $_POST['shipping_per_unit']): 0.0;
    $markupPercent     = isset($_POST['markup_percent'])   ? (float)str_replace(',', '.', $_POST['markup_percent'])   : 0.0;
    $markupFixed       = isset($_POST['markup_fixed'])     ? (float)str_replace(',', '.', $_POST['markup_fixed'])     : 0.0;

    $roundingStrategy  = $_POST['rounding_strategy'] ?? 'none';    // none|nearest|ceil|floor
    $roundingUnit      = (int)($_POST['rounding_unit'] ?? 1);      // 1|5|10
    if (!in_array($roundingStrategy, ['none','nearest','ceil','floor'], true)) $roundingStrategy = 'none';
    if (!in_array($roundingUnit, [1,5,10], true)) $roundingUnit = 1;

    $inputIncludesVat  = isset($_POST['input_includes_vat']) ? 1 : 0; // standard: av = netto
    $vatRate           = isset($_POST['vat_rate']) ? (float)str_replace(',', '.', $_POST['vat_rate']) : 25.0;
    $validUntilStr     = trim($_POST['valid_until'] ?? '');
    $validUntil        = $validUntilStr !== '' ? date('Y-m-d', strtotime($validUntilStr)) : null;

    // Fil
    if (!isset($_FILES['csv']) || $_FILES['csv']['error'] !== UPLOAD_ERR_OK) {
        $feedback[] = "Kunde inte läsa CSV-filen (upload error).";
    } else {
        $tmpPath  = $_FILES['csv']['tmp_name'];
        $origName = $_FILES['csv']['name'];

        $pdo = pdo();
        $pdo->beginTransaction();
        try {
            // import_runs: avrundningskolumner kan saknas i vissa installationer
            $haveRoundingCols = true;
            try { $pdo->query("SELECT rounding_strategy, rounding_unit FROM import_runs LIMIT 1"); }
            catch (Throwable $e) { $haveRoundingCols = false; }

            $sqlRun = "INSERT INTO import_runs
                (filename, discount_percent, discount_fixed_sek, markup_percent, markup_fixed_sek,
                 shipping_per_unit_sek, input_includes_vat, vat_rate_percent, valid_until";
            $sqlRunVals = ") VALUES (:f, :dp, :df, :mp, :mf, :spu, :iiv, :vat, :vu";
            if ($haveRoundingCols) { $sqlRun .= ", rounding_strategy, rounding_unit"; $sqlRunVals .= ", :rs, :ru"; }
            $sqlRun .= $sqlRunVals . ")";
            $stmtRun = $pdo->prepare($sqlRun);
            $paramsRun = [
                ':f'   => $origName,
                ':dp'  => $discountPercent,
                ':df'  => $discountFixed,
                ':mp'  => $markupPercent,
                ':mf'  => $markupFixed,
                ':spu' => $shippingPerUnit,
                ':iiv' => $inputIncludesVat,
                ':vat' => $vatRate,
                ':vu'  => $validUntil,
            ];
            if ($haveRoundingCols) {
                $paramsRun[':rs'] = $roundingStrategy;
                $paramsRun[':ru'] = $roundingUnit;
            }
            $stmtRun->execute($paramsRun);
            $importId = (int)$pdo->lastInsertId();

            $fh = fopen($tmpPath, 'r');
            if ($fh === false) throw new RuntimeException("Kunde inte öppna temporärfilen.");

            // Hoppa ev. BOM
            $firstBytes = fread($fh, 3);
            if ($firstBytes !== "\xEF\xBB\xBF") fseek($fh, 0);

            $delimiter = ';';
            $enclosure = '"';

            $headerFound = false;
            $headerMap   = [];
            $imported    = 0;
            $skipped     = 0;
            $skips       = new SkipLog();

            // ENDA prisfältet vi lagrar är price_net_sek
            $stmtUp = $pdo->prepare(
                "INSERT INTO refurb_products
                (sku, description, product_type, quantity,
                 price_net_sek, vat_rate_percent,
                 manufacturer, series, specification, grade,
                 valid_until, raw_row, import_id)
                VALUES
                (:sku, :description, :ptype, :qty,
                 :p_net, :vat,
                 :mfg, :series, :spec, :grade,
                 :vu, :raw, :imp)
                ON DUPLICATE KEY UPDATE
                  description=VALUES(description),
                  product_type=VALUES(product_type),
                  quantity=VALUES(quantity),
                  price_net_sek=VALUES(price_net_sek),
                  vat_rate_percent=VALUES(vat_rate_percent),
                  manufacturer=VALUES(manufacturer),
                  series=VALUES(series),
                  specification=VALUES(specification),
                  grade=VALUES(grade),
                  valid_until=VALUES(valid_until),
                  raw_row=VALUES(raw_row),
                  import_id=VALUES(import_id)"
            );

            while (($cols = fgetcsv($fh, 0, $delimiter, $enclosure)) !== false) {
                // Hitta rubriken
                if (!$headerFound) {
                    if (looksLikeHeader($cols)) {
                        $headerFound = true;
                        $headerMap   = mapHeader($cols);
                        if (!hasCoreFields($headerMap)) {
                            throw new RuntimeException("Rubrik hittad men nödvändiga kolumner saknas (kräver Nr, Beskrivning, Produkttyp, ÅF-Pris/AF-Pris, Tillverkare).");
                        }
                    }
                    continue;
                }

                // Tom/skräp
                $joined = normalizeSpaces(implode(' ', $cols));
                if ($joined === '') { $skipped++; $skips->add('tom rad', $cols); continue; }

                // Pris-kolumn
                $priceIdx = $headerMap['åf-pris'] ?? ($headerMap['af-pris'] ?? null);
                if ($priceIdx === null) { $skipped++; $skips->add('ingen pris-kolumn', $cols); continue; }

                // SKU: minst 4 tecken, behåll som sträng
                $skuRaw = trim((string)($cols[$headerMap['nr']] ?? ''));
                $skuRaw = preg_replace('/\s+/', '', $skuRaw);
                if (strlen($skuRaw) < 4) { $skipped++; $skips->add('ogiltig SKU (<4 tecken)', $cols); continue; }
                $sku = $skuRaw;

                // Grundfält
                $desc  = normalizeSpaces((string)($cols[$headerMap['beskrivning']] ?? ''));
                $ptype = normalizeSpaces((string)($cols[$headerMap['produkttyp']] ?? ''));
                $mfg   = normalizeSpaces((string)($cols[$headerMap['tillverkare']] ?? ''));
                if ($desc === '' || $ptype === '' || $mfg === '') { $skipped++; $skips->add('saknar beskrivning/typ/tillverkare', $cols); continue; }

                // Valfria
                $qtyCol = $headerMap['antal'] ?? null;
                $qty    = 0;
                if ($qtyCol !== null) { $qty = (int)preg_replace('/\D+/', '', (string)$cols[$qtyCol]); }

                $series = normalizeSpaces((string)($cols[$headerMap['serie'] ?? ''] ?? ''));
                $specIdx = $headerMap['specifikation'] ?? ($headerMap['specification'] ?? null);
                $spec  = $specIdx !== null ? normalizeSpaces((string)$cols[$specIdx]) : '';
                $grade = normalizeSpaces((string)($cols[$headerMap['grade'] ?? ''] ?? ''));

                // Pris från CSV
                $priceRaw = (string)$cols[$priceIdx];
                $priceSrcNorm = normalizePriceToDecimal($priceRaw);
                if ($priceSrcNorm === null) { $skipped++; $skips->add('pris ej tolkbart', $cols); continue; }
                $priceSrc = (float)$priceSrcNorm;

                // =======================
                // PRISLOGIK (netto som standard)
                // =======================
                $vatFactor = 1.0 + ($vatRate / 100.0);

                // 1) Grundnetto: standard = netto; om kryssat inkl. moms → konvertera till netto
                $baseNet = $inputIncludesVat ? ($priceSrc / $vatFactor) : $priceSrc;

                // 2) Rabatt (procent → fast)
                $netAfterDiscount = $baseNet * (1.0 - ($discountPercent / 100.0)) - $discountFixed;
                if ($netAfterDiscount < 0) $netAfterDiscount = 0.0;

                // 3) Frakt
                $netAfterShipping = $netAfterDiscount + $shippingPerUnit;
                if ($netAfterShipping < 0) $netAfterShipping = 0.0;

                // 4) Påslag (procent → fast)
                $netAfterMarkup = $netAfterShipping * (1.0 + ($markupPercent / 100.0)) + $markupFixed;
                if ($netAfterMarkup < 0) $netAfterMarkup = 0.0;

                // 5) Brutto
                $gross = $netAfterMarkup * $vatFactor;

                // 6) Avrunda BRUTTO
                $grossRounded = round_price($gross, $roundingStrategy, $roundingUnit);
                if ($grossRounded < 0) $grossRounded = 0.0;

                // 7) Räkna NETTO från avrundat BRUTTO (detta vi sparar)
                $netRounded = $grossRounded / $vatFactor;

                // Spara ENDAST NETTO
                $stmtUp->execute([
                    ':sku'      => $sku,
                    ':description' => $desc,
                    ':ptype'    => $ptype,
                    ':qty'      => $qty,
                    ':p_net'    => number_format($netRounded, 2, '.', ''),
                    ':vat'      => number_format($vatRate, 2, '.', ''),
                    ':mfg'      => $mfg,
                    ':series'   => $series,
                    ':spec'     => $spec,
                    ':grade'    => $grade,
                    ':vu'       => $validUntil,
                    ':raw'      => implode(';', array_map(fn($c) => str_replace(["\r","\n"], ' ', (string)$c), $cols)),
                    ':imp'      => $importId,
                ]);

                $imported++;
            }
            fclose($fh);
            $pdo->commit();

            $report = [
                'import_id' => $importId,
                'filename'  => $origName,
                'imported'  => $imported,
                'skipped'   => $skipped,
                'params'    => compact(
                    'discountPercent','discountFixed','shippingPerUnit',
                    'markupPercent','markupFixed',
                    'roundingStrategy','roundingUnit',
                    'inputIncludesVat','vatRate','validUntil'
                ),
                'skips'     => $debug ? $skips : null,
            ];

        } catch (Throwable $e) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            $feedback[] = "Fel vid import: " . htmlspecialchars($e->getMessage(), ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
        }
    }
}
?>
<!doctype html>
<html lang="sv">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Prisimport</title>
<style>
body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu,Cantarell,Helvetica,Arial,sans-serif;line-height:1.35;margin:2rem;background:#f6f7f9}
.container{max-width:860px;margin:0 auto;background:#fff;border-radius:14px;box-shadow:0 8px 24px rgba(0,0,0,.08);padding:1.2rem}
h1{font-size:1.4rem;margin:.2rem 0 1rem}
fieldset{border:1px solid #e2e8f0;border-radius:10px;margin:1rem 0;padding:1rem}
legend{padding:0 .5rem;color:#334155}
label{display:block;margin:.5rem 0 .25rem;font-weight:600}
input[type="text"],input[type="number"],input[type="date"],input[type="file"],select{
  width:100%;padding:.6rem .7rem;border:1px solid #cbd5e1;border-radius:10px;background:#fff}
.row{display:grid;grid-template-columns:1fr 1fr;gap:12px}
.btn{display:inline-block;background:#2563eb;color:#fff;border:0;border-radius:10px;padding:.7rem 1.1rem;font-weight:700;cursor:pointer}
.note{font-size:.9rem;color:#475569}
.alert{background:#fef3c7;border:1px solid #f59e0b;color:#92400e;padding:.7rem .9rem;border-radius:8px;margin:.5rem 0}
.success{background:#dcfce7;border:1px solid #22c55e;color:#14532d}
.table{width:100%;border-collapse:collapse;margin-top:1rem}
.table th,.table td{border:1px solid #e2e8f0;padding:.5rem .6rem;text-align:left;font-size:.95rem}
small{color:#64748b}
</style>
</head>
<body>
<div class="container">
  <h1>Importera prislista</h1>

  <?php foreach ($feedback as $f): ?>
    <div class="alert"><?= $f ?></div>
  <?php endforeach; ?>

  <?php if ($report): ?>
    <div class="alert success">
      <strong>Import klar!</strong><br>
      Import-ID: <b><?= (int)$report['import_id'] ?></b><br>
      Fil: <b><?= htmlspecialchars($report['filename'], ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') ?></b><br>
      Rader importerade: <b><?= (int)$report['imported'] ?></b><br>
      Rader hoppade över: <b><?= (int)$report['skipped'] ?></b>
    </div>
    <table class="table">
      <tr><th>Parameter</th><th>Värde</th></tr>
      <tr><td>Rabatt (%)</td><td><?= htmlspecialchars((string)$report['params']['discountPercent']) ?></td></tr>
      <tr><td>Rabatt fast (SEK)</td><td><?= htmlspecialchars((string)$report['params']['discountFixed']) ?></td></tr>
      <tr><td>Frakt per enhet (SEK)</td><td><?= htmlspecialchars((string)$report['params']['shippingPerUnit']) ?></td></tr>
      <tr><td><b>Påslag (%)</b></td><td><?= htmlspecialchars((string)$report['params']['markupPercent']) ?></td></tr>
      <tr><td><b>Påslag (SEK)</b></td><td><?= htmlspecialchars((string)$report['params']['markupFixed']) ?></td></tr>
      <tr><td>Avrundning</td><td><?= htmlspecialchars($report['params']['roundingStrategy'].' / '.$report['params']['roundingUnit'].' kr') ?></td></tr>
      <tr><td>CSV-priser är inklusive moms?</td><td><?= $report['params']['inputIncludesVat'] ? 'Ja' : 'Nej (standard)' ?></td></tr>
      <tr><td>Moms (%)</td><td><?= htmlspecialchars((string)$report['params']['vatRate']) ?></td></tr>
      <tr><td>Gäller till</td><td><?= htmlspecialchars((string)($report['params']['validUntil'] ?? '—')) ?></td></tr>
      <tr><td><b>Sparat pris</b></td><td><b>Endast NETTO (exkl. moms)</b></td></tr>
    </table>

    <?php if ($report['skips']): ?>
      <h3>Diagnostik (debug)</h3>
      <table class="table">
        <tr><th>Orsak</th><th>Antal</th><th>Exempel</th></tr>
        <?php foreach ($report['skips']->count as $reason => $cnt): ?>
          <tr>
            <td><?= htmlspecialchars($reason) ?></td>
            <td><?= (int)$cnt ?></td>
            <td>
              <?php foreach ($report['skips']->examples[$reason] as $ex): ?>
                <div><?= htmlspecialchars($ex) ?></div>
              <?php endforeach; ?>
            </td>
          </tr>
        <?php endforeach; ?>
      </table>
    <?php endif; ?>

    <hr>
  <?php endif; ?>

  <form method="post" enctype="multipart/form-data">
    <fieldset>
      <legend>CSV-fil</legend>
      <label>Prislista (CSV; semikolonseparerad)</label>
      <input type="file" name="csv" accept=".csv,text/csv" required>
      <p class="note">Sidan hittar rubrikraden (&quot;Nr;Beskrivning;Produkttyp;...&quot;) och hoppar över reklam i början. Lägg till <code>?debug=1</code> i URL för felsökningsstatistik.</p>
    </fieldset>

    <fieldset>
      <legend>Prissättning</legend>
      <div class="row">
        <div>
          <label>Rabatt i procent</label>
          <input type="number" name="discount_percent" step="0.001" min="0" value="0">
          <small>Ex: 10 betyder -10%.</small>
        </div>
        <div>
          <label>Rabatt fast belopp (SEK per enhet)</label>
          <input type="number" name="discount_fixed" step="0.01" min="0" value="0">
          <small>Dras av efter procent-rabatten.</small>
        </div>
      </div>

      <div class="row">
        <div>
          <label>Frakt per enhet (SEK)</label>
          <input type="number" name="shipping_per_unit" step="0.01" min="0" value="0">
        </div>
        <div>
          <label><b>Påslag i procent</b></label>
          <input type="number" name="markup_percent" step="0.001" value="0">
          <small>Läggs på efter frakt.</small>
        </div>
      </div>

      <div class="row">
        <div>
          <label><b>Påslag fast belopp (SEK per enhet)</b></label>
          <input type="number" name="markup_fixed" step="0.01" value="0">
        </div>
        <div>
          <label>Moms (%)</label>
          <input type="number" name="vat_rate" step="0.01" min="0" value="25.00">
          <small>Standard i Sverige: 25%.</small>
        </div>
      </div>

      <label><input type="checkbox" name="input_includes_vat"> Priser i CSV är <b>inklusive</b> moms</label>
      <p class="note">Om ej markerad behandlas CSV-priser som <b>exklusive</b> moms (standard).</p>
    </fieldset>

    <fieldset>
      <legend>Avrundning (på BRUTTO)</legend>
      <div class="row">
        <div>
          <label>Strategi</label>
          <select name="rounding_strategy">
            <option value="none">Ingen avrundning</option>
            <option value="nearest">Närmaste</option>
            <option value="ceil">Uppåt</option>
            <option value="floor">Nedåt</option>
          </select>
        </div>
        <div>
          <label>Steg (kr)</label>
          <select name="rounding_unit">
            <option value="1">1 kr</option>
            <option value="5">5 kr</option>
            <option value="10">10 kr</option>
          </select>
        </div>
      </div>
      <small>Avrundning sker på bruttopriset efter alla rabatter/frakt/påslag. Netto (som sparas) räknas om från det avrundade bruttopriset.</small>
    </fieldset>

    <fieldset>
      <legend>Giltighet</legend>
      <label>Priserna gäller till och med</label>
      <input type="date" name="valid_until" value="">
    </fieldset>

    <button class="btn" type="submit">Importera</button>
  </form>

  <p class="note" style="margin-top:1rem">
    Vi sparar endast <b>price_net_sek</b> (exkl. moms). SKU måste ha minst 4 tecken.
  </p>
</div>
</body>
</html>
